"""
This file contains common cost functions

@author: Stephan Bogs, Chair of Operations Management, RWTH Aachen
"""

import json

data = {}
factor = 1000000 # Normalize to million euro because of numerical stability


def refresh():
    with open('data/opt_params/params.json') as params_file:
        file_contents = params_file.read()
        data['params'] = json.loads(file_contents)

    # mass conversion factor from tons of biomass b to tons of biofuel with equal energy
    data['conv'] = data['params']["plants"]["conversion"]
    # periodic fixed operating costs for facility p for each point open in time t
    data['ffix'] = data['params']["plants"]["variable_operation"]
    # construction costs for facility p in case of new opening
    data['finv'] = data['params']["plants"]["investment"]
    # periodic variable operating costs for facility p for each point in time t
    data['frun'] = data['params']["plants"]["depreciation_years"]
    # heating value of biofuel in PJ/t
    data['HVbf'] = data['params']["pj_per_tonne_fuel"]
    data['tcrb'] = data['params']["transportation_cost"]["cost_biomass_t_km"]
    data['tcrf'] = data['params']["transportation_cost"]["cost_fuel_t_km"]
    # lifetime of the facility e.g. 25 years
    data['LT'] = data['params']["plants"]["depreciation_years"]
    # maximum capacity of the production facility
    data['sinv'] = data['params']["plants"]["output"]
    # uptime per period in which the facility is producing
    data['UT'] = data['params']["plants"]["operation_per_year"]
    # when do we reinvest
    data["reinvestment_period"] = data['params']["reinvestment_period"]
    # import from outside the system cost
    data["ofc"] = data["params"]["out_of_system_fuel_cost"]


refresh()


def get_transport_cost_from_solution(run, bio_scenario,  multi_scenario=False, demand_scenario=0):
    B = run["biomass_range"]
    I = run["regions_range"]
    T = run["years_range"]
    if multi_scenario:
        return sum(sum(
                (data['tcrb'] * run["distance_matrix"][i][p]) * run["xb1"][bio_scenario, demand_scenario, b, i, p, t] / run["heating_values"][b] / factor for b in B for i in I
                for p in I)
                    + sum((run["distance_matrix"][p][j] * data['tcrf']) * run["xf1"][bio_scenario, demand_scenario, p, j, t] / data['HVbf'] / factor for j in I for p in I) for t in
                    T)
    else:
        return sum(sum(
            (data['tcrb'] * run["distance_matrix"][i][p]) * run["xb"][b, i, p, t] / run["heating_values"][b] / factor for b in B for i in I
            for p in I)
                + sum((run["distance_matrix"][p][j] * data['tcrf']) * run["xf"][p, j, t] / data['HVbf'] / factor for j in I for p in I) for t in
                T)


def get_production_cost_from_solution(run, bio_scenario, multi_scenario=False, demand_scenario=0):
    I = run["regions_range"]
    T = run["years_range"]
    if multi_scenario:
        return (sum(data['frun'] * run["xf1"][bio_scenario, demand_scenario, p, j, t] / factor for j in I for p in I for t in T) 
            + sum((data['sinv'] * data['finv'] / data['LT'] + data['ffix']) * run["y0"][p] / factor for p in I for _ in range(data["reinvestment_period"]))
            + sum((data['sinv'] * data['finv'] / data['LT'] + data['ffix']) * run["y1"][bio_scenario,demand_scenario,p] /factor for p in I for _ in range(data["reinvestment_period"], T[-1] + 1))
            )
    else:
        return (sum(data['frun'] * run["xf"][p, j, t] / factor for j in I for p in I for t in T) 
            + sum((data['sinv'] * data['finv'] / data['LT'] + data['ffix']) * run["y0"][p] / factor for p in I for _ in range(data["reinvestment_period"]))
            + sum((data['sinv'] * data['finv'] / data['LT'] + data['ffix']) * run["y1"][p] / factor for p in I for _ in range(data["reinvestment_period"], T[-1] + 1))
            )


def get_biomass_cost_from_solution(run, bio_scenario, multi_scenario=False, demand_scenario=0):
    B = run["biomass_range"]
    I = run["regions_range"]
    T = run["years_range"]
    if multi_scenario:
        return sum(sum(run["biomass_cost"][bio_scenario][b][t][i] * 1000000 * run["xb1"][bio_scenario, demand_scenario, b, i, p, t] / factor for b in B for i in I for p in I) for t in T)
    else:
        return sum(sum(run["biomass_cost"][bio_scenario][b][t][i] * 1000000 * run["xb"][b, i, p, t] / factor for b in B for i in I for p in I) for t in T)
    
def get_outside_system_cost_from_solution(run, bio_scenario, multi_scenario=False, demand_scenario=0):
    I = run["regions_range"]
    T = run["years_range"]
    if multi_scenario:
        return sum(data["ofc"] * run['z'][bio_scenario,demand_scenario,j,t] / factor for j in I for t in T)
    else:
        return sum(data["ofc"] * run['z'][j,t] / factor for j in I for t in T)
    