"""
This file provides different ways of calculating the distance between two points given with their latitude and longitude

@author: Stephan Bogs, Chair of Operations Management, RWTH Aachen
"""

from math import sqrt, sin, cos, atan2, radians
import json

params = {}
with open('data/opt_params/params.json') as params_file:
    file_contents = params_file.read()
    params = json.loads(file_contents)


def manhatten_distance(loc1, loc2):
    """
    Calculates the Manhattan distance between two locations using the
    haversine distance for the delta-x and delta-y values
    """
    new_loc2 = {
        "lat": loc1["lat"],
        "lng": loc2["lng"]
    }
    x = compute_distance_hav(loc1, new_loc2)
    new_loc2 = {
        "lat": loc2["lat"],
        "lng": loc1["lng"]
    }
    y = compute_distance_hav(loc1, new_loc2)
    return x + y


def euclidian_distance(loc1, loc2):
    """
    Calculates the Manhattan distance between two locations using the
    haversine distance
    """
    return compute_distance_hav(loc1, loc2)


def compute_distance_hav(loc1, loc2):
    """
    This function uses the haversine to calculate the distance of two
    coordinates in kilometers.

    As the earth more or less a sphere (actually an elipsiod) we have to
    account for that when we calculate distances between two points. The
    haversine formular gives a good approximation, depicting the earth as a
    regular sphere.
    """
    lat1 = radians(loc1["lat"])
    lng1 = radians(loc1["lng"])
    lat2 = radians(loc2["lat"])
    lng2 = radians(loc2["lng"])

    dlng = lng2 - lng1
    dlat = lat2 - lat1

    a = sin(dlat / 2) ** 2 + cos(lat1) * cos(lat2) * sin(dlng / 2) ** 2
    c = 2 * atan2(sqrt(a), sqrt(1 - a))

    distance = 6378.140 * c

    return distance
